const Joi = require('joi');

const enquirySchema = Joi.object({
    fullname: Joi.string()
        .pattern(/^[a-zA-Z\s'-]+$/)
        .min(3)
        .max(50)
        .required()
        .messages({
            'string.base': 'Full name must be a text.',
            'string.empty': 'Full name is required.',
            'string.min': 'Full name should have at least 3 characters.',
            'string.max': 'Full name should have at most 50 characters.',
            'string.pattern.base': 'Full name can only contain letters, spaces, apostrophes, and hyphens.',
        }),
    email: Joi.string()
        .email()
        .required()
        .messages({
            'string.base': 'Email must be a text.',
            'string.email': 'Please provide a valid email address.',
            'string.empty': 'Email is required.',
        }),
    contact: Joi.string()
        .pattern(/^[789]\d{9}$/)
        .required()
        .messages({
            'string.base': 'Contact number must be a Number.',
            'string.empty': 'Contact number is required.',
            'string.pattern.base': 'Contact number must be a valid 10-digit number starting with 7, 8, or 9.',
        }),
    state: Joi.string()
        .min(2)
        .max(50)
        .required()
        .messages({
            'string.base': 'State must be a text.',
            'string.empty': 'State is required.',
            'string.min': 'State should have at least 2 characters.',
            'string.max': 'State should have at most 50 characters.',
        }),
    city: Joi.string()
        .min(2)
        .max(50)
        .required()
        .messages({
            'string.base': 'City must be a text.',
            'string.empty': 'City is required.',
            'string.min': 'City should have at least 2 characters.',
            'string.max': 'City should have at most 50 characters.',
        }),
    investment: Joi.number()
        .min(1)
        .required()
        .messages({
            'number.base': 'Investment must be a number.',
            'number.empty': 'Investment is required.',
            'number.min': 'Investment should be at least 1.',
        }),
    message: Joi.string()
        .min(2)
        .max(500)
        .required()
        .messages({
            'string.base': 'Message must be a text.',
            'string.empty': 'Message is required.',
            'string.min': 'Message should have at least 5 characters.',
            'string.max': 'Message should have at most 500 characters.',
        }),
    source: Joi.string().required().messages({
        'string.base': 'Source must be a text.',
        'string.empty': 'Source is required.',
    }),    
    created_at: Joi.date()
        .default(() => new Date())
        .messages({
            'date.base': 'Created date must be a valid date.',
        }),
});


const contactSchema = Joi.object({
    name: Joi.string().min(3).max(30).required().messages({
        'string.base': 'Name must be a text.',
        'string.empty': 'Name is required.',
        'string.min': 'Name should have at least 3 characters.',
        'string.max': 'Name should have at most 30 characters.',
    }),
    email: Joi.string()
        .email()
        .required()
        .messages({
            'string.base': 'Email must be a text.',
            'string.email': 'Please provide a valid email address.',
            'string.empty': 'Email is required.',
        }),
    mobile: Joi.string()
        .pattern(/^[789]\d{9}$/)
        .required()
        .messages({
            'string.base': 'Mobile number must be a text.',
            'string.empty': 'Mobile number is required.',
            'string.pattern.base': 'Mobile number must be a valid 10-digit number starting with 7, 8, or 9.',
        }),
    created_at: Joi.date()
        .default(() => new Date())
        .messages({
            'date.base': 'Created date must be a valid date.',
        }),
});


function validateEnquiry(req, res, next) {
    const { error } = enquirySchema.validate(req.body);
    
    if (error) {
        return res.status(400).json({
            status: "error",
            message: error.details.map((err) => err.message),
        });
    }
    next();
}


function validateContact(req, res, next) {
    const { error } = contactSchema.validate(req.body);

    if (error) {
        return res.status(400).json({
            status: "error",
            message: error.details.map((err) => err.message),
        });
    }
    next();
}

module.exports = {
    validateEnquiry,
    validateContact,
};
